;*****************************************************************************        
;
;   Module:     ui.inc
;               
;   Author:     Mike Hibbett 
;                                                                  
;   Version:    1.0 5/10/03                                                  
;
;               Functions to support implemention of the user interface 
;
;*****************************************************************************        



;*****************************************************************************        
;
;   Function :  uiDspBCDWord
;               prints a 6 digit BCD number to the LCD at the current cursor
;               position. 
;
;   Input:      MSB pointed to by INDF. Bytes must be in sequence
;
;   Output:     Display updated
;
;*****************************************************************************        
uiDspBCDWord
    call    uiDspBCDByte
    call    uiDspBCDByte
    call    uiDspBCDByte
    return



;*****************************************************************************        
;
;   Function :  uiDspBCDByte
;               prints a 2 digit BCD number to the LCD at the current cursor
;               position. 
;
;   Input:      BCD byte pointed to by INDF. 
;
;   Output:     Display updated. INDF incremented
;
;*****************************************************************************        
uiDspBCDByte
    swapf   INDF, W
    andlw   0x0F
    addlw   '0'
    call    dspPutChar
    movfw   INDF
    andlw   0x0F
    addlw   '0'
    call    dspPutChar
    incf    FSR, F
    return



;*****************************************************************************        
;
;   Function :  uiDspDistance
;               Displays the appropriate distance unit on the LCD 
;
;   Input:      The eeprom byte contain the units flags must have be read
;
;   Output:     None
;
;*****************************************************************************        
uiDspDistance
    movlw   st1m6 - st1m1       ; Miles
    btfss   eepData, 0
    movlw   st1m7 - st1m1       ; KM
    call    uiDspStr1
    return
    


;*****************************************************************************        
;
;   Function :  uiDspVolume
;               Displays the appropriate fuel volume unit on the LCD 
;
;   Input:      The eeprom byte contain the units flags must have be read
;
;   Output:     None
;
;*****************************************************************************        
uiDspVolume
    movlw   st1m5 - st1m1       ; Gallons
    btfss   eepData, 1
    movlw   st1m4 - st1m1       ; Litres
    call    uiDspStr1
    return



;*****************************************************************************        
;
;   Function :  kbdInit
;               Configures the hardware to accept data from a keypad
;               This is mostly interrupt definitions and pullup settings
;               RB4 - 7 are input, pulled up by internal pullup resistors
;               RB0 - 3 are output, driven low, not pulled up! 
;               PortB is not shared with any other device, so we have full 
;               control 
;
;   Input:      None
;
;   Output:     None
;
;*****************************************************************************        
kbdInit
    bsf     STATUS, RP0    
    movlw   0x0F0                   ; 1 == input
    movwf   TRISB
    bcf     OPTION_REG, NOT_RBPU    ; Enable portB pullups
    bcf     STATUS, RP0    
    clrf    PORTB
    return



;*****************************************************************************        
;
;   Function : kbdWaitNoKey
;               Waits for any pressed keys to be released
;
;   Input:      None
;
;   Output:     None
;
;*****************************************************************************        
kbdWaitNoKey
    ; wait for any depressed keys to be released
    call    kbdReadRaw
    movfw   rawKey
    sublw   0x0f
    btfss   STATUS, Z
    goto    kbdWaitNoKey
  
    movlw   D'1'
    call    uiWait10ms

    call    kbdReadRaw
    movfw   rawKey
    sublw   0x0f
    btfss   STATUS, Z
    goto    kbdWaitNoKey
    return



;*****************************************************************************        
;
;   Function :  kbdDebounce
;               delays for 10ms then polls the keyboard again to confirm
;               consistent key value
;              
;               Key code return in rawKey, 0x0F == no key
;              
;   Input:      Raw keycode in rawKey
;
;   Output:     confirmed rawKeycode
;
;*****************************************************************************        
kbdDebounce
    movfw   rawKey
    movwf   keyTmp
    ; Wait 10ms, then check that the key is the same value
    movlw   D'1'
    call    uiWait10ms
    call    kbdReadRaw

    movfw   rawKey
    subwf   keyTmp,w
    btfss   STATUS, Z
    goto    kbdDebounce
    return



;*****************************************************************************        
;
;   Function :  kbdReadRaw
;               Reads the keyboard, determining the current keycode
;              
;               Key code return in rawKey, 0x0F == no key
;              
;               This does no debouncing; Debouncing is done elsewhere.
;               Key code conversion is also done elsewhere
;
;   Input:      None
;
;   Output:     Raw key code return in rawKey, 0x0F == no key
;
;*****************************************************************************        
kbdReadRaw
    swapf   PORTB, W                ; get the input bits in LSB
    andlw   0x00F                   ; This will be 0x0F if no key pressed
    movwf   rawKey
    movwf   keyTmp1
    sublw   0x00F
    btfsc   STATUS, Z               ; If it was 0x0f, we can exit
    return
    
    clrf    rawKey
    btfss   keyTmp1, 0
    goto    krr000
    incf    rawKey,F
    btfss   keyTmp1, 1
    goto    krr000
    incf    rawKey,F
    btfss   keyTmp1, 2
    goto    krr000
    incf    rawKey,F
             
krr000    
    ; We have a key pressed; Don't know which row yet though.
    movlw   0x001
    movwf   PORTB
    nop
    swapf   PORTB, W                ; get the keys back
    andlw   0x00F
    sublw   0x00F
    btfss   STATUS, Z               
    goto    krr001                  ; Not this one
    clrf    PORTB
    return

krr001
    movlw   0x002
    movwf   PORTB
    nop
    swapf   PORTB, W                ; get the keys back
    andlw   0x00F
    sublw   0x00F
    btfss   STATUS, Z               
    goto    krr002                  ; Not this one
    movlw   0x004                   ; this is the one
    addwf   rawKey, F
    clrf    PORTB
    return

krr002
    movlw   0x004
    movwf   PORTB
    nop
    swapf   PORTB, W                ; get the keys back
    andlw   0x00F
    sublw   0x00F
    btfss   STATUS, Z               
    goto    krr003                  ; Not this one
    movlw   0x008                   ; this is the one
    addwf   rawKey, F
    clrf    PORTB
    return

krr003
    movlw   0x0F
    movwf    rawKey
    clrf    PORTB
    return



;*****************************************************************************        
;
;   Function :  kbdReadKey
;               Reads the keyboard, waiting for a key to be pressed
;              
;               Key code return in rawKey (Converted to ASCII)
;              
;   Input:      None
;
;   Output:     ASCII code in rawKey and W
;
;*****************************************************************************        
kbdReadKey
    call    kbdWaitNoKey
    
getKey
    ; Wait for a key to be pressed 
    call    kbdReadRaw
    movfw   rawKey
    sublw   0x0f
    btfsc   STATUS, Z
    goto    getKey

    call    kbdDebounce
    movfw   rawKey
    sublw   0x0f
    btfsc   STATUS, Z
    goto    getKey

    call    kbdConvert
    movwf   rawKey
    return



;*****************************************************************************        
;
;   Function :  uiMenu
;               (uiWelcome)
;               Displays the menu as two lines.
;               The display is assumed to be already switched on
;               NOTE: dspWelcome is a 'special' entry point for displaying
;               the welcome screen. 
;
;   Input:      menu to display on first line in menuItem (uiMenu only) 0..MENUSIZE
;
;   Output:     Display updated.
;
;*****************************************************************************        
uiMenu
    movfw   menuItem
    movwf   menuLine
    sublw   MENU_SIZE - 1
    btfss   STATUS, C
    clrf    menuLine            ; There was an overflow, so wrap round
    btfss   STATUS, C
    clrf    menuItem            ; Clear the input parameter too
    call    dspClear
       
    incf    menuLine, F
    incf    menuLine, F         ; Skip first two lines - these are the 
                                   ; Welcome screen
       
    movlw   0x07e
    call    dspPutChar          ; Show prompt

    movlw   D'15'               ; We can only show 15 on the first line
    goto    dsp001
       
uiWelcome                      ; Entry point for welcome screen
    clrf    menuLine   
    movlw   D'16'
    
dsp001
    movwf   DMCount
       
    swapf   menuLine, W         ; menuline x 16 = eeprom address
    bsf     STATUS, RP1            ; Select bank 2 
    movwf   EEADR
    
dmread
    bsf     STATUS, RP1            ; Select bank 2 
    bsf     STATUS, RP0            ; Select bank 3
    bcf     EECON1, EEPGD
    bsf     EECON1, RD
    bcf     STATUS, RP0            ; Select bank 2
    movfw   EEDATA
    incf    EEADR, F            ; Move to next menu byte
    bcf     STATUS, RP1
    call    dspPutChar    
    decfsz  DMCount, F
    goto    dmread

    movfw   menuLine
    sublw   MENU_SIZE + 1
    btfsc   STATUS, Z
    return                      ; Display only 1 line if at end of list
    
    call    dspLine2            ; Move to start of next line

    movlw   D'16'              
    movwf   DMCount

    swapf   menuLine, W         
    addlw   D'16'

    bsf     STATUS, RP1            ; Select bank 2 
    movwf   EEADR
    
dmread2
    bsf     STATUS, RP1            ; Select bank 2 
    bsf     STATUS, RP0            ; Select bank 3
    bcf     EECON1, EEPGD
    bsf     EECON1, RD
    bcf     STATUS, RP0            ; Select bank 2
    movfw   EEDATA
    incf    EEADR, F            ; Move to next menu byte
    bcf     STATUS, RP1
    call    dspPutChar    

    decfsz  DMCount, F
    goto    dmread2
    return



;*****************************************************************************        
;
;   Function : uiWait100us
;              delays for a multiple of 100us
;
;   Input:     multiple in W
;
;*****************************************************************************        
uiWait100us
    movwf   delay2

d1us002
    movlw    D'165'                    
    movwf    delay1                   
    
d1us001    
    decfsz    delay1, F               
    goto    d1us001                    
    decfsz  delay2, F
    goto    d1us002
    return                         
    

    
;*****************************************************************************        
;
;   Function : uiWait10ms
;              delays for a multiple of 10ms
;
;   Input:     multiple in W
;
;*****************************************************************************        
uiWait10ms
    movwf   delay3
    
d1ms001
    movlw   D'100'
    call    uiWait100us
    decfsz  delay3, F
    goto    d1ms001
    return



;*****************************************************************************        
;
;   Function :  kbdConvert
;               Converts the raw key code in rawKey into a real key
;               in W. Value is 0x00 if no key pressed
;
;   Input:      key in rawKey
;
;   Output:     real key ASCII value in W
;
;*****************************************************************************        
kbdConvert
    movlw   high kbdConvert
    movwf   PCLATH
    movfw   rawKey
    andlw   0x0f
    addwf    PCL ,F
    retlw   '#'
    retlw   '9'
    retlw   '6'
    retlw   '3'
    retlw   '0'
    retlw   '8'
    retlw   '5'
    retlw   '2'
    retlw   '*'
    retlw   '7'
    retlw   '4'
    retlw   '1'
    retlw   ' '
    retlw   ' '
    retlw   ' '
kbdConvert_end
    retlw   ' '
    
    IF ( (kbdConvert & 0x0FF) >= (kbdConvert_end & 0x0FF) )
        MESSG   "Table kbdConvert overflow"
    ENDIF



;*****************************************************************************        
;
;   Function :  uiDspStr1
;               Prints a string to the display at the current cursor 
;               position
;
;   Input:      string to display in W
;
;   Output:     Display modified
;
;*****************************************************************************        
uiDspStr1
    movwf    DataPointer    
        
uips1_001
    movlw   HIGH strTable1
    movwf   PCLATH    
    movf    DataPointer, W
    call    strTable1
    clrf    PCLATH
    xorlw   0
    btfsc   STATUS, Z
    goto    uips1_002
    call    dspPutChar
    incf    DataPointer, F
    goto    uips1_001

uips1_002
    clrf    PCLATH
    return



 ORG 0x800



strTable1
    addwf   PCL, F
st1m1
    retlw    'D'
    retlw    'D'
    retlw    '/'
    retlw    'M'
    retlw    'M'
    retlw    '/'
    retlw    'Y'
    retlw    'Y'
    retlw    ' '
    retlw    ' '
    retlw    'H'
    retlw    'H'
    retlw    ':'
    retlw    'M'
    retlw    'M'
    retlw    0
st1m2
    retlw    'P' 
    retlw    'r'
    retlw    'e'
    retlw    's'
    retlw    's'
    retlw    ' '
    retlw    'a'
    retlw    'n'
    retlw    'y'
    retlw    ' '
    retlw    'k'
    retlw    'e'
    retlw    'y'
    retlw    0
st1m3
    retlw    't'
    retlw    'o'
    retlw    ' '
    retlw    'c'
    retlw    'l'
    retlw    'o'
    retlw    's' 
    retlw    'e'
    retlw    ' '
    retlw    'P'
    retlw    'C'
    retlw    ' '
    retlw    'l'
    retlw    'i'
    retlw    'n'
    retlw    'k'
    retlw    0
st1m4
    retlw    'L'
    retlw    'i'
    retlw    't'
    retlw    'r'
    retlw    'e'
    retlw    0
st1m5
    retlw    'G'
    retlw    'a'
    retlw    'l'
    retlw    'l'
    retlw    'o'
    retlw    'n'
    retlw    0  
st1m6
    retlw    'M'
    retlw    'i'
    retlw    'l'
    retlw    'e'
    retlw    's'
    retlw    0  
st1m7
    retlw    'K'
    retlw    'M'
    retlw    0  
st1m8
    retlw    '*'
    retlw    '='
    retlw    'N'
    retlw    'e'
    retlw    'x'
    retlw    't'
    retlw    ','
    retlw    ' '
    retlw    '#'
    retlw    '='
    retlw    'D'
    retlw    'o'
    retlw    'n'
    retlw    'e'
    retlw    0  
st1m9
    retlw    '*'
    retlw    '='
    retlw    'D'
    retlw    'e'
    retlw    'l'
    retlw    'e'
    retlw    't'
    retlw    'e'
    retlw    ','
    retlw    ' '
    retlw    '#'
    retlw    '='
    retlw    'D'
    retlw    'o'
    retlw    'n'
    retlw    'e'
    retlw    0  
st1m10
    retlw    '*'
    retlw    '='
    retlw    'C'
    retlw    'h'
    retlw    'a'
    retlw    'n'
    retlw    'g'
    retlw    'e'
    retlw    ','
    retlw    ' '
    retlw    '#'
    retlw    '='
    retlw    'D'
    retlw    'o'
    retlw    'n'
    retlw    'e'
    retlw    0  
st1m11
    retlw    '*'
    retlw    '='
    retlw    'C'
    retlw    'a'
    retlw    'n'
    retlw    'c'
    retlw    'e'
    retlw    'l'
    retlw    ','
    retlw    ' '
    retlw    '#'
    retlw    '='
    retlw    'Y'
    retlw    'e'
    retlw    's'
    retlw    0  
st1m12
    retlw    'C'
    retlw    'l'
    retlw    'e'
    retlw    'a'
    retlw    'r'
    retlw    ' '
    retlw    'M'
    retlw    'e'
    retlw    'm'
    retlw    'o'
    retlw    'r'
    retlw    'y'
    retlw    '?'
    retlw    0  
st1m13
    retlw    'M'
    retlw    'e'
    retlw    'm'
    retlw    'o'
    retlw    'r'
    retlw    'y'
    retlw    ' '
    retlw    'F'
    retlw    'u'
    retlw    'l'
    retlw    'l'
    retlw    0  
st1m14                  ; For selecting reminder type
    retlw    'T'
    retlw    'y'
    retlw    'p'
    retlw    'e'
    retlw    ':'
    retlw    ' '
    retlw    0  
st1m15
    retlw    'D'
    retlw    'i'
    retlw    's'
    retlw    't'
    retlw    'a'
    retlw    'n'
    retlw    'c'
    retlw    'e'
    retlw    0  
st1m16
    retlw    'D'
    retlw    'a'
    retlw    't'
    retlw    'e'
    retlw    0  
st1m17
    retlw    'D'
    retlw    'i'
    retlw    's'
    retlw    't'
    retlw    '/'
    retlw    'D'
    retlw    'a'
    retlw    't'
    retlw    'e'
    retlw    0  
st1m18
    retlw    'N'
    retlw    'o'
    retlw    ' '
    retlw    'e'
    retlw    'n'
    retlw    't'
    retlw    'r'
    retlw    'i'
    retlw    'e'
    retlw    's'
    retlw    0  
st1m19
    retlw    'D'
    retlw    'D'
    retlw    '/'
    retlw    'M'
    retlw    'M'
    retlw    '/'
    retlw    'Y'
    retlw    'Y'
    retlw    0  
st1m20
    retlw    '+'
    retlw    'M'
    retlw    'i'
    retlw    'n'
    retlw    '/'
    retlw    'M'
    retlw    'o'
    retlw    'n'
    retlw    't'
    retlw    'h'
    retlw    ' '
    
strTable1_end
    retlw    0
    
    IF ( (strTable1 & 0x0FF) >= (strTable1_end & 0x0FF) )
        MESSG   "Table strTable1 overflow"
    ENDIF



